/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  Codec,
  buildCodecForObject,
  canonicalizeBaseUrl,
  codecForString,
  codecOptional
} from "@gnu-taler/taler-util";

export interface KycUiSettings {
  // Where exchange backend is localted
  // default: window.origin without "webui/"
  backendBaseURL?: string;

}

/**
 * Global settings for the bank UI.
 */
const defaultSettings: KycUiSettings = {
  backendBaseURL: buildDefaultBackendBaseURL(),
};

const codecForKycUiSettings = (): Codec<KycUiSettings> =>
  buildCodecForObject<KycUiSettings>()
    .property("backendBaseURL", codecOptional(codecForString()))
    .build("codecForKycUiSettings");

function removeUndefineField<T extends object>(obj: T): T {
  const keys = Object.keys(obj) as Array<keyof T>;
  return keys.reduce((prev, cur) => {
    if (typeof prev[cur] === "undefined") {
      delete prev[cur];
    }
    return prev;
  }, obj);
}

export function fetchSettings(listener: (s: KycUiSettings) => void): void {
  fetch("./settings.json")
    .then((resp) => resp.json())
    .then((json) => codecForKycUiSettings().decode(json))
    .then((result) =>
      listener({
        ...defaultSettings,
        ...removeUndefineField(result),
      }),
    )
    .catch((e) => {
      console.log("failed to fetch settings", e);
      listener(defaultSettings);
    });
}

function buildDefaultBackendBaseURL(): string | undefined {
  if (typeof window !== "undefined") {
    const currentLocation = new URL(
      window.location.pathname,
      window.location.origin,
    ).href;
    /**
     * By default, bank backend serves the html content
     * from the /webui root.
     */
    return canonicalizeBaseUrl(currentLocation.replace("/webui", ""));
  }
  throw Error("No default URL");
}
