% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seq.R
\name{seq}
\alias{seq}
\alias{seq.tind}
\title{Create a Sequence of Time Indices}
\usage{
\method{seq}{tind}(from, to, by = 1, length.out = NULL, along.with = NULL, ...)
}
\arguments{
\item{from}{an object of \code{tind} class or an R object coercible to it.}

\item{to}{an object of \code{tind} class or an R object coercible to it.}

\item{by}{a numeric value, a \code{tdiff}, or a character string determining increment.}

\item{length.out}{an integer value, the desired length.}

\item{along.with}{any R object, length of this argument will be taken as the desired length.}

\item{...}{(ignored) further arguments passed to or from other methods.}
}
\value{
An object of \code{tind} class.
}
\description{
\code{seq} method for objects of \code{tind} class allows to easily construct
sequences of time indices of all supported types.
}
\details{
\code{seq} method requires that exactly two of the three arguments \code{from},
\code{to}, and \code{length.out} are provided. If \code{along.with} is not
\code{NULL}, its length is used as value of \code{length.out}.

\code{by} can be a number, an object of \code{tdiff} class (of length 1), or an object
coercible to \code{tdiff} like \code{"3w"} denoting step by three weeks.
\code{by} cannot be \code{NA} and cannot be 0 when both \code{from} and \code{to} are
provided. Given both \code{from} and \code{to}, sign of \code{by}
has to agree with the order of \code{from} and \code{to}.
When \code{by} is a number, the underlying unit of time is assumed. For time of day
and date-time indices this is always a second.

\code{from} and \code{to} can be of different types provided that conversion is possible.
The result is of higher resolution. This allows to easily construct series like
from begging of the month to today, from today till the end of next year, etc.
See Examples.

\code{seq.tind} slightly differs from \code{\link[base]{seq.Date}}
in terms of interface and requirements with respect to arguments.
Firstly, \code{from} argument can be missing (provided that \code{to} and
\code{length.out} are given). Secondly, \code{by} has the default value of 1.

Both \code{seq} method for \code{tind} and \code{seq.tind} function are
exported allowing for conversion to \code{tind} as in
\code{seq.tind("2025-01", "2025-12")}.
}
\examples{
# sequences of dates by 1 and 2 months from now
(td <- today())
seq(td, by = "1m", length.out = 12)
seq(td, by = "2m", length.out = 6)
# sequences of dates by 1 and 2 months to now
seq(to = td, by = "1m", length.out = 12)
seq(to = td, by = "2m", length.out = 6)
# sequence of dates from the beginning of the month till today
seq(floor_t(td, "m"), td)
# same
seq(as.month(td), td)
# sequence of dates from today till the end of the next month
seq(td, as.month(td) + 1)
# sequence of date-time from now to midnight by 1 hour
(nw <- now())
seq(nw, ceiling_t(nw, "1d"), by = "1h")
# same
seq(nw, as.date(nw), by = "1h")
# sequence (date-time) from full hour to now by 2 minutes
seq(floor_t(nw, "1h"), nw, by = "2min")
# sequence (time of day) from full hour to now by 2 minutes
seq(floor_t(as.time(nw), "1h"), as.time(nw), by = "2min")
# sequence (date-time) from now down to full hour by 2 minutes
seq(nw, floor_t(nw, "1h"), by = "-2min")
# sequence (time of day) from now down to full hour by 2 minutes
seq(as.time(nw), floor_t(as.time(nw), "1h"), by = "-2min")
# sequence (date-time) of length 10 from now down by 10 seconds
seq(nw, by = -10, length.out = 10)
# sequence (time of day) of length 10 from now down by 10 seconds
seq(as.time(nw), by = -10, length.out = 10)
# explicit call to seq.tind with conversion
seq.tind("2025-01", "2025-12")
## corner cases
# from 2025-12-30 23:00 till end of 2025, note that 2025-12-31 24:00
# (that is 2025-01-01 00:00) is excluded from the result as it is in the next year
seq(as.tind("2025-12-30 23:00", tz = "UTC"), "2025", by = "5h")
# from end of 2025 down to 2025-12-30 23:00, note that 2025-12-31 24:00
# (that is 2025-01-01 00:00) is excluded from the result as it is in the next year
seq(as.tind("2025"), as.tind("2025-12-30 23:00", tz = "UTC"), by = "-5h")

}
