% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_cube.R
\name{sits_cube}
\alias{sits_cube}
\title{Create data cubes from image collections}
\usage{
sits_cube(source, collection, ...)
}
\arguments{
\item{source}{Data source: one of \code{"AWS"}, \code{"BDC"},
\code{"CDSE"}, \code{"DEAFRICA"}, \code{"DEAUSTRALIA"},
\code{"HLS"}, \code{"PLANETSCOPE"}, \code{"MPC"},
\code{"SDC"} or \code{"USGS"}.}

\item{collection}{Image collection in data source.
To find out the supported collections,
use \code{\link{sits_list_collections}()}).}

\item{...}{Other parameters to be passed for specific types.}
}
\value{
A \code{tibble} describing the contents of a data cube.
}
\description{
Creates a data cube based on spatial and temporal restrictions
in collections available in cloud services or local repositories.
Available options are:
\itemize{
\item{To create data cubes from providers which support the STAC protocol,
use \code{\link[sits]{sits_cube.stac_cube}}.}
\item{To create raster data cubes from local image files,
use \code{\link[sits]{sits_cube.local_cube}}.}
\item{To create vector data cubes from local image and vector files,
use \code{\link[sits]{sits_cube.vector_cube}}.}
\item{To create raster data cubes from local image files
which have been classified or labelled,
use \code{\link[sits]{sits_cube.results_cube}}.}
}
}
\note{
The main \code{sits} classification workflow has the following steps:
\enumerate{
     \item{\code{\link[sits]{sits_cube}}: selects a ARD image collection from
         a cloud provider.}
     \item{\code{\link[sits]{sits_cube_copy}}: copies an ARD image collection
         from a cloud provider to a local directory for faster processing.}
     \item{\code{\link[sits]{sits_regularize}}: create a regular data cube
         from an ARD image collection.}
     \item{\code{\link[sits]{sits_apply}}: create new indices by combining
         bands of a  regular data cube (optional).}
     \item{\code{\link[sits]{sits_get_data}}: extract time series
         from a regular data cube based on user-provided labelled samples.}
     \item{\code{\link[sits]{sits_train}}: train a machine learning
         model based on image time series.}
     \item{\code{\link[sits]{sits_classify}}: classify a data cube
         using a machine learning model and obtain a probability cube.}
     \item{\code{\link[sits]{sits_smooth}}: post-process a probability cube
         using a spatial smoother to remove outliers and
         increase spatial consistency.}
     \item{\code{\link[sits]{sits_label_classification}}: produce a
         classified map by selecting the label with the highest probability
         from a smoothed cube.}
}

The following cloud providers are supported, based on the STAC protocol:
Amazon Web Services (AWS), Brazil Data Cube (BDC),
Copernicus Data Space Ecosystem (CDSE), Digital Earth Africa (DEAFRICA),
Digital Earth Australia (DEAUSTRALIA), Microsoft Planetary Computer (MPC),
Nasa Harmonized Landsat/Sentinel (HLS), Swiss Data Cube (SDC), TERRASCOPE and
USGS Landsat (USGS). Data cubes can also be created using local files.

In \code{sits}, a data cube is represented as a tibble with metadata
describing a set of image files obtained from cloud providers.
It contains information about each individual file.

A data cube in \code{sits} is:
\itemize{
\item{A set of images organized in tiles of a grid system (e.g., MGRS).}
\item{Each tile contains single-band images in a
 unique zone of the coordinate system (e.g, tile 20LMR in MGRS grid)
 covering the period between \code{start_date} and \code{end_date}.}
\item{Each image of a tile is associated to a unique temporal interval.
All intervals share the same spectral bands.}
\item{Different tiles may cover different zones of the same grid system.}
}
A regular data cube is a data cube where:
\itemize{
\item{All tiles share the same set of regular temporal intervals.}
\item{All tiles share the same spectral bands and indices.}
\item{All images have the same spatial resolution.}
\item{Each location in a tile is associated a set of multi-band time series.}
\item{For each tile, interval and band, the cube is reduce to a 2D image.}
}
}
\examples{
if (sits_run_examples()) {
    # --- Access to the Brazil Data Cube
    # create a raster cube file based on the information in the BDC
    cbers_tile <- sits_cube(
        source = "BDC",
        collection = "CBERS-WFI-16D",
        bands = c("NDVI", "EVI"),
        tiles = "007004",
        start_date = "2018-09-01",
        end_date = "2019-08-28"
    )
    # --- Access to Digital Earth Africa
    # create a raster cube file based on the information about the files
    # DEAFRICA does not support definition of tiles
    cube_deafrica <- sits_cube(
        source = "DEAFRICA",
        collection = "SENTINEL-2-L2A",
        bands = c("B04", "B08"),
        roi = c(
            "lat_min" = 17.379,
            "lon_min" = 1.1573,
            "lat_max" = 17.410,
            "lon_max" = 1.1910
        ),
        start_date = "2019-01-01",
        end_date = "2019-10-28"
    )
    # --- Create a cube based on a local MODIS data
    # MODIS local files have names such as
    # "TERRA_MODIS_012010_NDVI_2013-09-14.jp2"
    # see the parse info parameter as an example on how to
    # decode local files
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    modis_cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6.1",
        data_dir = data_dir,
        parse_info = c("satellite", "sensor", "tile", "band", "date")
    )
}
}
\author{
Felipe Carlos, \email{efelipecarlos@gmail.com}

Felipe Carvalho, \email{felipe.carvalho@inpe.br}

Gilberto Camara, \email{gilberto.camara@inpe.br}

Rolf Simoes, \email{rolfsimoes@gmail.com}
}
