% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/srt_ranks.r
\name{srt_ranks}
\alias{srt_ranks}
\title{Ranks for the signed-rank test.}
\usage{
srt_ranks(x, call, warn_wd = FALSE)
}
\arguments{
\item{x}{(named list)\cr
A named list which contains a vector of paired differences named \code{"diffs"}.
Typically the object returned by \code{\link[=srt_data]{srt_data()}}.}

\item{call}{(call or named list)\cr
A call or named list which contains the arguments from the parent function \code{\link[=srt]{srt()}} or \code{\link[=rdt]{rdt()}}.
The minimum list of arguments required for \code{srt_ranks()} are:
\itemize{
\item \code{mu}
\itemize{
\item Scalar numeric \verb{(-Inf, Inf)}.
Under the null hypothesis, \code{diffs} is assumed to be symmetric around \code{mu}.
}
\item \code{zero_method}
\itemize{
\item Scalar character \code{c("wilcoxon", "pratt")}.
String for zero handling.
}
\item \code{digits_rank}
\itemize{
\item Scalar numeric \verb{(0, Inf]}.
Controls ranking precision.
}
}

Next in line, \code{\link[=srt_method]{srt_method()}} needs:
\itemize{
\item \verb{[[1L]]}
\itemize{
\item \code{as.name()} of calling function
\item One of \code{quote(srt)}, \code{quote(srt2)}, \code{quote(rdt)}, or \code{quote(rdt2)}
}
\item \code{distribution}
\itemize{
\item Scalar character: \code{c("auto", "exact", "asymptotic")}
\item The method used to calculate the p-value.
}
}}

\item{warn_wd}{(Scalar logical: \code{c(FALSE, TRUE)})\cr
Used for
\enumerate{
\item Midpoint algorithm 'W+(d)' in confidence interval test inversion.
\item Recalculating ranks after removing mu-shift.
If \code{TRUE} return a warning and exit with value \code{wplus = 0}.
}}
}
\value{
list
}
\description{
Computes ranks of the absolute differences and the signed-rank test statistic \eqn{W^+}.
The returned list is designed to be reused by higher-level signed-rank functions.
}
\details{
Consider a numeric vector of paired differences \eqn{x = (x_1, \dots, x_n)}.
After removing \code{NA} values, let \eqn{\mathcal{I}_0 = \{i : x_i = 0\}} and \eqn{\mathcal{I}_{\neq 0} = \{i : x_i \neq 0\}}.
\subsection{Zero handling}{

For the Wilcoxon method (\code{zero_method = "wilcoxon"}), zeros are removed prior to ranking, so the ranking is performed on \eqn{\{x_i : i \in \mathcal{I}_{\neq 0}\}} only.

For the Pratt method (\code{zero_method = "pratt"}), zeros are retained when computing ranks, but their corresponding ranks do not contribute to the signed-rank sum \eqn{W^+}.
}

\subsection{Ranking}{

Ranks are assigned to \eqn{|x_i|} using average-tie handling.
The argument \code{digits_rank} controls rounding for ranking only.
Ranks are computed from \eqn{|\mathrm{signif}(x_i, \text{digits\_rank})|} when \code{digits_rank} is finite, and from \eqn{|x_i|} otherwise.
This rounding may induce ties, which changes both the values of the averaged ranks and the variance of the statistic in asymptotic procedures.

The next function in the 'srt pipeline', \code{\link[=srt_method]{srt_method()}}, calculates the number of ties among the ranks (\code{n_ties}).
}

\subsection{Wilcoxon signed-Rank statistic}{

Let \eqn{r_i} denote the absolute-value rank of the \eqn{i}-th observation after applying the chosen zero handling and ranking precision.
The sum of positive ranks is

\deqn{W^+ = \sum_{i : x_i > 0} r_i,}

which is the canonical Wilcoxon signed-rank statistic used for both exact and asymptotic inference.
}
}
\examples{
library(rankdifferencetest)

# Synthetic paired differences with zeros and ties
set.seed(1)
diffs <- c(rnorm(8, mean = 0.3), 0, 0, 0, round(rnorm(8, mean = -0.2), 1))
x <- list(diffs = diffs)

# Wilcoxon zero method: zeros dropped before ranking
call <- list(mu = 0, zero_method = "wilcoxon", digits_rank = Inf)
cw <- rankdifferencetest:::srt_ranks(x, call)
cw$ranks
cw$wplus

# Pratt zero method: zeros retained for ranking
call <- list(mu = 0, zero_method = "pratt", digits_rank = Inf)
cp <- rankdifferencetest:::srt_ranks(x, call)
cp$wplus
cp$n_signed

# Induce ties via ranking precision
call <- list(mu = 0, zero_method = "wilcoxon", digits_rank = 1)
ctied <- rankdifferencetest:::srt_ranks(x, call)
ctied$ranks

}
\keyword{internal}
