% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv.R
\name{cv.pgee}
\alias{cv.pgee}
\title{Cross validation for Penalized Generalized Estimating Equations}
\usage{
cv.pgee(N, m, X, Z = NULL, y = NULL, yc = NULL, yb = NULL, K = 5,
  grid1, grid2 = NULL, wctype = "Ind", family = "Gaussian", eps = 1e-06,
  maxiter = 1000, tol.coef = 0.001, tol.score = 0.001, init = NULL,
  standardize = TRUE, penalty = "SCAD", warm = TRUE, weights = rep(1,
  N), type_c = "square", type_b = "deviance", marginal = 0, FDR = FALSE,
  fdr.corr = NULL, fdr.type = "all")
}
\arguments{
\item{N}{Number of clusters.}

\item{m}{Cluster size. Assumed equal across all clusters. Should be set to 2
for family=="Mixed".}

\item{X}{Design matrix. If family=="Mixed", then design matrix for continuous
responses. For family!="Mixed", should have N*m rows. For family=="Mixed",
should have N rows.}

\item{Z}{Design matrix for binary responses for family=="Mixed". Should not
be provided for other family types. If not provided for family=="Mixed", is
set equal to X. For family!="Mixed", should have N*m rows. For
family=="Mixed", should have N rows.}

\item{y}{Response vector. Don't use this argument for family == "Mixed".
Instead, use arguments yc and yb. Since the cluster size is assumed equal
across clusters, the vector is assumed to have the form c(y_1,
y_2,...,y_N), with y_i = c(y_i1,...,y_im).}

\item{yc}{Continuous response vector. Use only for family=="Mixed".}

\item{yb}{Binary (0/1) response vector. Use only for family=="Mixed".}

\item{K}{Number of folds.}

\item{grid1}{For family!="Mixed", the grid of tuning parameters. For
family=="Mixed", the grid of tuning parameters for coefficients
corresponding to the continuous outcomes.}

\item{grid2}{For family=="Mixed", the grid of tuning parameters for
coefficients corresponding to the binary outcomes. Not used for
family!="Mixed".}

\item{wctype}{Working correlation type; one of "Ind", "CS", or "AR1". For
family=="Mixed", "CS" and "AR1" are equivalent.}

\item{family}{"Gaussian", "Binomial", or "Mixed" (use the last for bivariate
mixed outcomes). Note that for "Binomial", currently only binary outcomes
are supported.}

\item{eps}{Disturbance in the Linear Quadratic Approximation algorithm.}

\item{maxiter}{The maximum number of iterations the Newton algorithm tries
before declaring failure to converge.}

\item{tol.coef}{Converge of the Newton algorithm is declared if two
conditions are met: The L1-norm of the difference of successive iterates
should be less than tol.coef AND the L1-norm of the penalized score should
be less than tol.score.}

\item{tol.score}{See \code{tol.coef}.}

\item{init}{Vector of initial values for regression coefficients. For
family=="Mixed", should be c(init_c, init_b). Defaults to glm values.}

\item{standardize}{Standardize the design matrices prior to estimation?}

\item{penalty}{"SCAD", "MCP", or "LASSO".}

\item{warm}{Use warm starts?}

\item{weights}{Vector of cluster weights. All observations in a cluster are
assumed to have the same weight.}

\item{type_c}{Loss function for continuous outcomes. "square" (square error
loss, the default) or "absolute" (absolute error loss).}

\item{type_b}{Loss function for binary outcomes. "deviance" (binomial
deviance, the default) or "classification" (prediction error).}

\item{marginal}{For the mixed outcomes case, set to 0 (the default) to
account for both the continuous loss and the binary loss, set to 1 to only
account for the continuous loss, and set to 2 to only account for the
binary loss.}

\item{FDR}{Should the false discovery rate be estimated for family=="Mixed"?
Currently, FDR cannot be estimated for other family types.}

\item{fdr.corr}{Association parameter to use in FDR estimation. The default
is to use the association parameter estimated from the PGEEs.}

\item{fdr.type}{Estimate the FDR for only the coefficients corresponding to
the continuous outcomes ("continuous"), for only the coefficients
corresponding to the binary outcomes ("binary"), or for all coefficients
("all", the default).}
}
\value{
A list
  \item{coefficients}{Vector of estimated regression
  coefficients. For family=="Mixed", this takes the form c(coef_c, coef_b).}
  \item{vcov}{Sandwich formula based covariance matrix of estimated
  regression coefficients (other than the intercept(s)). The row/column
  names correspond to elements of \code{coefficients}.}
  \item{phi}{Estimated dispersion parameter.}
  \item{alpha}{Estimated association parameter.}
  \item{num_iterations}{Number of iterations the Newton algorithm ran.}
  \item{converge}{0=converged, 1=did not converge.}
  \item{PenScore}{Vector of penalized score functions at the
  estimated regression coefficients. If the algorithm converges, then these
  should be close to zero.}
  \item{FDR}{Estimated FDR for family=="Mixed", if requested.}
  \item{lambda.loss}{Cross validation loss (error) for the
  optimal tuning parameter(s) lambda, averaged across folds.}
  \item{LossMat}{Matrix of cross validation losses. Rows denote tuning
  parameter values, columns denote folds.}
}
\description{
Performs k-fold cross-validation for Penalized Generalized Estimating
Equations (PGEEs) over grid(s) of tuning parameters lambda. Linear and binary
logistic models are supported. In particular, can handle the case of
bivariate correlated mixed outcomes, in which each cluster consists of one
continuous outcome and one binary outcome.
}
\details{
The function calls \code{pgee.fit} \code{K} times, each time leaving out
1/\code{K} of the data. The cross-validation error is determined by the
arguments \code{type_c} and \code{type_b}. For family=="Mixed", the
cross-validation error is (by default) the sum of the continuous error and
the binary error.
}
\examples{
\dontrun{
# Gaussian
N <- 100
m <- 10
p <- 50
y <- rnorm(N * m)
# If you want standardize = TRUE, you must provide an intercept.
X <- cbind(1, matrix(rnorm(N * m * (p - 1)), N * m, p - 1))
gr1 <- seq(0.001, 0.1, length.out = 100)
fit <- cv.pgee(X = X, y = y, N = N, m = m, grid1 = gr1, wctype = "CS",
            family = "Gaussian")

# Binary
y <- sample(0:1, N*m, replace = TRUE)
fit <- cv.pgee(X = X, y = y, N = N, m = m, grid1 = gr1, wctype = "CS",
            family = "Binomial")

# Bivariate mixed outcomes
# Generate some data
Bc <- c(2.0, 3.0, 1.5, 2.0, rep(0,times=p-4))
Bb <- c(0.7, -0.7, -0.4, rep(0,times=p-3))
dat <- gen_mixed_data(Bc, Bc, N, 0.5)
# We require two grids of tuning parameters
gr2 <- seq(0.0001, 0.01, length.out = 100)
# Estimate regression coefficients and false discovery rate
fit <- cv.pgee(X = dat$X, Z = dat$Z, yc = dat$yc, yb = dat$yb, N = N, m = 2,
               wctype = "CS", family = "Mixed", grid1 = gr1, grid2 = gr2,
               FDR = TRUE)
}
}

