#' Fits the linear mixed model to estimate the dependent ICCs
#'
#' @keywords internal
#' @param dataset A data frame with (at least) three columns: the outcome, the subject and the setting identifiers.
#' @param ry Character string. The outcome variable.
#' @param rind Character string. The subject identifier.
#' @param rtype Character string. The setting identifier.
#' @return An object of class lme.
fit_model_dep_icc<-function(dataset,ry,rind,rtype,warnings=TRUE){
  
  dades<- data.frame(dataset) |> dplyr::select(any_of(c(ry,rind,rtype)))
  
  dades <- dades |> dplyr::rename(y = all_of(ry),
                                  ind = all_of(rind),
                                  type = all_of(rtype)) %>% 
    mutate(ind=factor(ind),type=factor(type))
  
  # Model with different ICCs
  model_dif<-try(lme(y~type,random = ~-1+type|ind,
                     weights=varIdent(form=~1|type),
                     data=dades,na.action="na.omit"),silent=T)
  
  # Change optimizer in case of no convergence
  if(class(model_dif)[1]=='try-error'){
    
    model_dif<-try(lme(y~type,random = ~-1+type|ind,
                       weights=varIdent(form=~1|type),data=dades,
                       control=list(opt="optim"),na.action="na.omit"))
  }
  
  
  # Checking and fixing error in variances
  
  if(class(model_dif)[1]!='try-error'){
    if(class(model_dif$apVar)[1]=="character"){
      
      if (warnings) message("Error on apVar")
      model_dif<-tryCatch(lme(y~type,random = ~-1+type|ind,
                              weights=varIdent(form=~1|type),
                              data=dades,
                              control=list(opt="optim",minAbsParApVar=0.1)),
                          error=function(e){})
      
      if(class(model_dif)[1]=='try-error'){
        model_dif<-NULL
        if (warnings) message("Model did not converge")
      }
    }
  }
  
  return(model_dif)
}
