% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/randSVD.R
\name{randSVD}
\alias{randSVD}
\title{Randomized Singular Value Decomposition.}
\usage{
randSVD(A, k = NULL, l = NULL, its = 2, sdist = "unif")
}
\arguments{
\item{A}{array_like \cr
a real/complex input matrix (or data frame), with dimensions
\eqn{(m, n)}. It is the real/complex matrix being approximated.}

\item{k}{int, optional \cr
determines the target rank of the low-rank decomposition and should
satisfy \eqn{k << min(m,n)}. Set by default to 6.}

\item{l}{int, optional \cr
block size of the block Lanczos iterations; \eqn{l} must be a
positive integer greater than \eqn{k}, and defaults \eqn{l=k+2}.}

\item{its}{int, optional \cr
number of full iterations of a block Lanczos method to conduct;
\eqn{its} must be a nonnegative integer, and defaults to 2.}

\item{sdist}{str \eqn{c('normal', 'unif')}, optional \cr
Specifies the sampling distribution. \cr
\eqn{'unif'} : (default) Uniform `[-1,1]`. \cr
\eqn{'normal}' : Normal `~N(0,1)`. \cr}
}
\value{
\code{randSVD} returns a list containing the following three components:
\item{d}{
          array_like \cr
          \eqn{(k,k)} matrix in the rank-k approximation USV' to A, where A is
          \eqn{(m,n)}; the entries of \eqn{S} are all nonnegative, and its only
          nonzero entries appear in nonincreasing order on the diagonal.
}

\item{u}{
          matrix \cr
          \eqn{(m, k)} matrix in the rank-\eqn{k} approximation \eqn{A = U * diag(S) * V'}
          to A; the columns of U are orthonormal and are called Left singular vect.
          We want to remark that this is the transpose matrix, hence
          the vectors are on the rows of our matrix.
}

\item{v}{
          matrix \cr
          \eqn{(n, k)} matrix in the rank-\eqn{k} approximation \eqn{A = U * diag(S) * V'}
          to A; the columns of V are orthonormal and are called Right singular vect.
}
}
\description{
Compute the near-optimal low-rank singular value decomposition (SVD) of
a rectangular matrix. The algorithm follows a randomized approach.
}
\details{
Randomized SVD (randSVD) is a fast algorithm to compute the approximate low-rank SVD of
a rectangular \eqn{(m,n)} matrix \eqn{A} using a probabilistic algorithm.
Given the decided rank \eqn{k << n}, \code{rSVD} factors the input matrix \eqn{A} as
\eqn{A = U * diag(S) * V'}, which is the typical SVD form. Precisely, the columns of
U are orthonormal, as are the columns of V, the entries of S are all nonnegative,
and the only nonzero entries of S appear in non-increasing order on its diagonal. The
dimensions are:  U is \eqn{(m,k)}, V is \eqn{(n,k)}, and S is \eqn{(k,k)}, when A
is \eqn{(m,n)}.

Increasing \eqn{its} or \eqn{l} improves the accuracy of the approximation USV' to A.

The parameter \eqn{its} specifies the number of normalized power iterations
(subspace iterations) to reduce the approximation error. This is recommended
if the the singular values decay slowly. In practice 1 or 2 iterations
achieve good results, however, computing power iterations increases the
computational time. The number of power iterations is set to \eqn{its=2} by default.
}
\note{
The singular vectors are not unique and only defined up to sign
(a constant of modulus one in the complex case). If a left singular vector
has its sign changed, changing the sign of the corresponding right vector
gives an equivalent decomposition.
}
\examples{
#Simulate a general matrix with 1000 rows and 1000 columns
vy= rnorm(1000*1000,0,1)
y= matrix(vy,1000,1000,byrow=TRUE)

#Compute the randSVD for the first hundred components of the matrix y and measure the time
start.time <- Sys.time()
prova1= randSVD(y,k=100)
Sys.time()- start.time

#Compare with a classical SVD
start.time <- Sys.time()
prova2= svd(y)
Sys.time()- start.time


}
\references{
\itemize{
  \item  [1] N. Halko, P. Martinsson, and J. Tropp.\cr
         "Finding structure with randomness: probabilistic
         algorithms for constructing approximate matrix
         decompositions" (2009). \cr
         (available at arXiv \url{http://arxiv.org/abs/0909.4061}).
  \item  [2] S. Voronin and P.Martinsson.\cr
         "RSVDPACK: Subroutines for computing partial singular value
         decompositions via randomized sampling on single core, multi core,
         and GPU architectures" (2015).\cr
         (available at `arXiv \url{http://arxiv.org/abs/1502.05366}).
 \item   [3] N. Benjamin Erichson.\cr
         "Randomized Singular Value Decomposition (rsvd): R package" (2016).\cr
         (available in the CRAN).
 \item   [4] Nathan Halko, Per-Gunnar Martinsson, and Joel Tropp.\cr
         "Finding structure with randomness: Stochastic algorithms for
         constructing approximate matrix decompositions" (2009).\cr
         (available at \url{http://arxiv.org}).
 \item   [5] V. Rokhlin, A. Szlam, M. Tygert.\cr
         "A randomized algorithm for principal component analysis" (2009).\cr
         (available at \url{https://arxiv.org/abs/0809.2274}).\cr
         The implementation of rand SVD is inspired by the MatLab implementation
         of RandPCA by M. Tygert.
}
}
\author{
L. Rimella, \email{lorenzo.rimella@hotmail.it}
}
