% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_data.R
\name{get_data_by_dims}
\alias{get_data_by_dims}
\title{Extract Data by Dimension Patterns from SL4 or HAR Objects}
\usage{
get_data_by_dims(
  patterns = NULL,
  ...,
  experiment_names = NULL,
  subtotal_level = FALSE,
  rename_cols = NULL,
  merge_data = FALSE,
  pattern_mix = FALSE
)
}
\arguments{
\item{patterns}{Character vector. Dimension patterns to extract. Use \code{"ALL"} or \code{NULL} to extract all available patterns.}

\item{...}{One or more SL4 or HAR data objects loaded using \code{load_sl4x()} or \code{load_harx()}.}

\item{experiment_names}{Character vector. Names assigned to each dataset. If \code{NULL}, names are inferred.}

\item{subtotal_level}{Character or logical. Determines which decomposition levels to retain:
\itemize{
\item \code{"total"}: Keeps only \code{"TOTAL"} values.
\item \code{"decomposed"}: Keeps only decomposed values (excludes \code{"TOTAL"}).
\item \code{"all"}: Keeps all rows.
\item \code{TRUE}: Equivalent to \code{"all"}, retaining both \code{"TOTAL"} and decomposed values.
\item \code{FALSE}: Equivalent to \code{"total"}, keeping only \code{"TOTAL"} values.
}}

\item{rename_cols}{Named vector. Column name replacements (\code{c("old_name" = "new_name")}).}

\item{merge_data}{Logical. If \code{TRUE}, attempts to merge data across multiple experiments. Default is \code{FALSE}.}

\item{pattern_mix}{Logical. If \code{TRUE}, allows flexible pattern matching, ignoring dimension order. Default is \code{FALSE}.}
}
\value{
A structured list of extracted data:
\itemize{
\item If \code{merge_data = FALSE}, returns a named list where each element corresponds to an experiment.
\item If \code{merge_data = TRUE}, returns a named list of all merged data
}
}
\description{
Retrieves structured data from SL4 or HAR objects based on specified dimension patterns.
Supports multiple experiments and merging datasets while maintaining structured dimension metadata.
}
\details{
\itemize{
\item Extracts variables matching specified dimension patterns.
\item Allows for flexible pattern matching (\code{pattern_mix = TRUE}).
\item Supports merging data across multiple experiments (\code{merge_data = TRUE}).
\item Provides column renaming functionality (\code{rename_cols}).
\item Handles subtotal filtering (\code{subtotal_level}), controlling whether \code{"TOTAL"} or decomposed values are retained.
}
}
\examples{
# Import sample data:
sl4_data <- load_sl4x(
  system.file("extdata", "TAR10.sl4", package = "HARplus")
)
sl4_data1 <- load_sl4x(
  system.file("extdata", "SUBT10.sl4", package = "HARplus")
)

# Extract data for a single dimension pattern
data_single_pattern <- get_data_by_dims(
  "comm*reg",
  sl4_data
)

# Extract multiple dimension patterns
data_multiple_patterns <- get_data_by_dims(
  c("comm*reg", "REG*ACTS"),
  sl4_data
)

# Extract all dimension patterns separately from multiple datasets
data_all_patterns <- get_data_by_dims(
  NULL,
  sl4_data, sl4_data1,
  merge_data = FALSE
)

# Merge data for identical patterns across multiple datasets
data_merged_patterns <- get_data_by_dims(
  NULL,
  sl4_data, sl4_data1,
  merge_data = TRUE
)

# Merge data while allowing interchangeable dimensions (e.g., A*B = B*A)
data_pattern_mixed <- get_data_by_dims(
  NULL,
  sl4_data, sl4_data1,
  merge_data = TRUE,
  pattern_mix = TRUE
)

# Retain only "TOTAL" values
data_total_only <- get_data_by_dims(
  "comm*reg",
  sl4_data,
  subtotal_level = "total"
)
data_total_only_alt <- get_data_by_dims(
  "comm*reg",
  sl4_data,
  subtotal_level = FALSE
)

# Retain only decomposed components
data_decomposed_only <- get_data_by_dims(
  "comm*reg",
  sl4_data,
  subtotal_level = "decomposed"
)

# Retain all value levels
data_all_decomp <- get_data_by_dims(
  "comm*reg",
  sl4_data,
  subtotal_level = "all"
)
data_all_decomp_alt <- get_data_by_dims(
  "comm*reg",
  sl4_data,
  subtotal_level = TRUE
)

# Rename specific columns
data_renamed <- get_data_by_dims(
  "comm*reg",
  sl4_data,
  rename_cols = c(REG = "Region", COMM = "Commodity")
)

# Merge data with custom experiment names
data_merged_experiments <- get_data_by_dims(
  "comm*reg",
  sl4_data, sl4_data1,
  experiment_names = c("EXP1", "EXP2"),
  merge_data = TRUE
)

}
\seealso{
\code{\link{get_data_by_var}}, \code{\link{group_data_by_dims}}
}
\author{
Pattawee Puangchit
}
