% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdSolve_hdf5}
\alias{bdSolve_hdf5}
\title{Solve Linear System AX = B (HDF5-Stored)}
\usage{
bdSolve_hdf5(
  filename,
  groupA,
  datasetA,
  groupB,
  datasetB,
  outgroup = NULL,
  outdataset = NULL,
  overwrite = NULL
)
}
\arguments{
\item{filename}{String. Path to the HDF5 file.}

\item{groupA}{String. Group containing matrix A.}

\item{datasetA}{String. Dataset name for matrix A.}

\item{groupB}{String. Group containing matrix B.}

\item{datasetB}{String. Dataset name for matrix B.}

\item{outgroup}{Optional string. Output group name (defaults to "Solved").}

\item{outdataset}{Optional string. Output dataset name (defaults to "A_B").}

\item{overwrite}{Logical. Whether to overwrite existing results.}
}
\value{
List with components. If an error occurs, all string values are returned as empty strings (""):
\describe{
\item{fn}{Character string with the HDF5 filename}
\item{ds}{Character string with the full dataset path to the solution of the linear system (group/dataset)}
}
}
\description{
Solves the linear system AX = B where matrices A and B are stored in HDF5
format. The solution X is written back to the HDF5 file.
}
\details{
This function provides an HDF5-based implementation for solving large linear
systems. Key features:
\itemize{
\item HDF5 Integration:
\itemize{
\item Efficient reading of input matrices
\item Memory-efficient processing
\item Direct output to HDF5 format
}
\item Implementation Features:
\itemize{
\item Automatic solver selection
\item Support for large matrices
\item Flexible output options
\item Memory-efficient processing
}
}

The function handles:
\itemize{
\item Data validation
\item Memory management
\item Error handling
\item HDF5 file operations
}
}
\examples{
library(BigDataStatMeth)

# Create test matrices
N <- 1000
M <- 1000
fn <- "test_temp.hdf5"

set.seed(555)
Y <- matrix(rnorm(N*M), N, M)
X <- matrix(rnorm(N), N, 1)
Ycp <- crossprod(Y)

# Compare with in-memory solution
resm <- bdSolve(Ycp, X)
resr <- solve(Ycp, X)
all.equal(resm, resr)

# Save matrices to HDF5
bdCreate_hdf5_matrix(filename = fn,
                     object = Ycp,
                     group = "data",
                     dataset = "A",
                     transp = FALSE,
                     overwriteFile = TRUE,
                     overwriteDataset = TRUE,
                     unlimited = FALSE)

bdCreate_hdf5_matrix(filename = fn,
                     object = X,
                     group = "data",
                     dataset = "B",
                     transp = FALSE,
                     overwriteFile = FALSE,
                     overwriteDataset = TRUE,
                     unlimited = FALSE)

# Solve using HDF5-stored matrices
bdSolve_hdf5(filename = fn,
             groupA = "data",
             datasetA = "A",
             groupB = "data",
             datasetB = "B",
             outgroup = "Solved",
             outdataset = "A_B",
             overwrite = TRUE)

# Cleanup
if (file.exists(fn)) {
    file.remove(fn)
}

}
\references{
\itemize{
\item Anderson, E. et al. (1999). LAPACK Users' Guide, 3rd Edition.
SIAM, Philadelphia.
\item The HDF Group. (2000-2010). HDF5 User's Guide.
}
}
\seealso{
\itemize{
\item \code{\link{bdSolve}} for in-memory matrix solving
\item \code{\link{bdCreate_hdf5_matrix}} for creating HDF5 matrices
}
}
