/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#if defined(_WIN32)

static inline herr_t
H5TS_semaphore_signal(H5TS_semaphore_t *sem)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(0 == ReleaseSemaphore(*sem, 1, NULL)))
        return FAIL;

    return SUCCEED;
} 

static inline herr_t
H5TS_semaphore_wait(H5TS_semaphore_t *sem)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(WAIT_OBJECT_0 != WaitForSingleObject(*sem, INFINITE)))
        return FAIL;

    return SUCCEED;
} 

#elif defined(__unix__) && !defined(__MACH__)

static inline herr_t
H5TS_semaphore_signal(H5TS_semaphore_t *sem)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(0 != sem_post(sem)))
        return FAIL;

    return SUCCEED;
} 

static inline herr_t
H5TS_semaphore_wait(H5TS_semaphore_t *sem)
{
    int rc;

    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    
    do {
        rc = sem_wait(sem);
    } while (rc == -1 && errno == EINTR);

    if (H5_UNLIKELY(0 != rc))
        return FAIL;

    return SUCCEED;
} 
#else

static inline herr_t
H5TS_semaphore_signal(H5TS_semaphore_t *sem)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&sem->mutex) < 0))
        return FAIL;

    
    if (sem->waiters)
        if (H5_UNLIKELY(H5TS_cond_signal(&sem->cond) < 0)) {
            H5TS_mutex_unlock(&sem->mutex);
            return FAIL;
        }

    
    sem->counter++;

    
    if (H5_UNLIKELY(H5TS_mutex_unlock(&sem->mutex) < 0))
        return FAIL;

    return SUCCEED;
} 

static inline herr_t
H5TS_semaphore_wait(H5TS_semaphore_t *sem)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&sem->mutex) < 0))
        return FAIL;

    
    while (0 == sem->counter) {
        herr_t ret;

        
        sem->waiters++;
        ret = H5TS_cond_wait(&sem->cond, &sem->mutex);
        sem->waiters--;

        
        if (H5_UNLIKELY(ret < 0)) {
            H5TS_mutex_unlock(&sem->mutex);
            return FAIL;
        }
    }

    
    sem->counter--;

    
    if (H5_UNLIKELY(H5TS_mutex_unlock(&sem->mutex) < 0))
        return FAIL;

    return SUCCEED;
} 
#endif
