## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 5
)

library(gvcAnalyzer)

## -----------------------------------------------------------------------------
io <- bm_build_io(
  Z = bm_toy_Z,
  Y = bm_toy_Y,
  VA = bm_toy_VA,
  X = bm_toy_X,
  countries = bm_toy_countries,
  sectors = bm_toy_sectors
)

## -----------------------------------------------------------------------------
out_comp <- bm_2025_output_components(io)
out_comp

## -----------------------------------------------------------------------------
out_meas <- bm_2025_output_measures(io)
out_meas

## -----------------------------------------------------------------------------
# Compute sectoral components and measures
out_comp_sec <- bm_2025_output_components_sector(io)
out_meas_sec <- bm_2025_output_measures_sector(io)
head(out_meas_sec, 12)

## -----------------------------------------------------------------------------
# Example: Compare manufacturing sectors across countries
# Note: Using column names specific to the sectoral function (X_i, share_GVC_output_i, etc.)
manufacturing <- out_meas_sec[out_meas_sec$sector == "Manufacturing", ]

# Select key columns for display
cols_to_show <- c("country", "sector", "X_i", "share_GVC_output_i", "forward_output_i")
manufacturing[, cols_to_show]

## -----------------------------------------------------------------------------
oldpar <- par(mar = c(5, 5, 3, 2))

barplot(
  out_meas$share_GVC_output,
  names.arg = out_meas$country,
  col = "steelblue",
  ylab = "GVC Share of Output",
  main = "Output-Based GVC Participation",
  ylim = c(0, max(out_meas$share_GVC_output, na.rm = TRUE) * 1.2)
)

grid()

par(oldpar)

## -----------------------------------------------------------------------------
oldpar <- par(mar = c(5, 5, 3, 2))

composition <- as.matrix(out_meas[, c("share_PF_output", "share_TS_output", "share_PB_output")])
rownames(composition) <- out_meas$country

barplot(
  t(composition),
  beside = FALSE,
  col = c("darkgreen", "orange", "darkred"),
  ylab = "Share of GVC Output",
  main = "GVC Output Composition",
  legend.text = c("Pure-Forward", "Two-Sided", "Pure-Backward"),
  args.legend = list(x = "topright", bty = "n")
)

grid()

par(oldpar)

## -----------------------------------------------------------------------------
oldpar <- par(mar = c(5, 5, 3, 2))

barplot(
  out_meas$forward_output,
  names.arg = out_meas$country,
  col = ifelse(out_meas$forward_output > 0, "darkgreen", "darkred"),
  ylab = "Forward Orientation Index",
  main = "Output-Based Forward Orientation",
  ylim = c(-1, 1)
)

abline(h = 0, lty = 2, col = "gray", lwd = 2)

grid()

par(oldpar)

## -----------------------------------------------------------------------------
# Aggregate sector-level results to country level
# Note: Using X_i and GVC_Xi for sector-level columns
sec_agg <- aggregate(
  cbind(X_i, GVC_Xi) ~ country,
  data = out_comp_sec,
  FUN = sum
)

# Calculate implied country share from sector sums
sec_agg$share_GVC_output <- sec_agg$GVC_Xi / sec_agg$X_i

# Compare with direct country-level calculation
# CORRECTED: Using out_meas (which has the shares) instead of out_comp
comparison <- merge(
  out_meas[, c("country", "share_GVC_output")],
  sec_agg[, c("country", "share_GVC_output")],
  by = "country",
  suffixes = c("_direct", "_sectoral")
)

comparison

## -----------------------------------------------------------------------------
sessionInfo()

