#!/usr/bin/perl
use v5.14;
use warnings;
use HiPi::Wiring qw( :wiring );
use HiPi::Device::SPI;

use constant SPI_DEV   => '/dev/spidev0.0';
use constant SPI_SPEED => 1_000_000;
use constant LED       => 14;
use constant POWER     => 2;
use constant SCE       => 8;
use constant RST       => 24;
use constant DC        => 23;
my @PIC_OUT = (
    0xFF, 0x8D, 0x9F, 0x13, 0x13, 0xF3, 0x01, 0x01, 0xF9, 0xF9, 0x01, 0x81, 0xF9,
    0xF9, 0x01, 0xF1, 0xF9, 0x09, 0x09, 0xFF, 0xFF, 0xF1, 0xF9, 0x09, 0x09, 0xF9,
    0xF1, 0x01, 0x01, 0x01, 0x01, 0x01, 0xF9, 0xF9, 0x09, 0xF9, 0x09, 0xF9, 0xF1,
    0x01, 0xC1, 0xE9, 0x29, 0x29, 0xF9, 0xF1, 0x01, 0xFF, 0xFF, 0x71, 0xD9, 0x01,
    0x01, 0xF1, 0xF9, 0x29, 0x29, 0xB9, 0xB1, 0x01, 0x01, 0x01, 0xF1, 0xF1, 0x11,
    0xF1, 0xF1, 0xF1, 0xE1, 0x01, 0xE1, 0xF1, 0x51, 0x51, 0x71, 0x61, 0x01, 0x01,
    0xC1, 0xF1, 0x31, 0x31, 0xF1, 0xFF, 0xFF, 0x00, 0x01, 0x01, 0x01, 0x01, 0x60,
    0xE0, 0xA0, 0x01, 0x01, 0x81, 0xE1, 0x61, 0x60, 0xC0, 0x01, 0xE1, 0xE1, 0x21,
    0x21, 0xE0, 0xC1, 0x01, 0xC1, 0xE1, 0x20, 0x20, 0xFC, 0xFC, 0xE0, 0xE0, 0xC1,
    0xE1, 0xE0, 0xC1, 0xE0, 0xE1, 0x01, 0xFC, 0xFC, 0x21, 0x21, 0xE1, 0xC1, 0xE5,
    0xE4, 0x01, 0xC1, 0xE0, 0x20, 0x21, 0x20, 0x00, 0x01, 0xFD, 0xFD, 0x21, 0x20,
    0xE0, 0x00, 0x00, 0x01, 0x01, 0xC0, 0x61, 0x31, 0x31, 0x21, 0x20, 0xC0, 0x81,
    0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0xFF, 0xFF,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x02, 0x03, 0x01, 0x00, 0x01, 0x03, 0xF2,
    0x1A, 0x0B, 0x08, 0x0B, 0x1B, 0x10, 0x60, 0xE3, 0x03, 0x00, 0x01, 0x03, 0x02,
    0x02, 0x03, 0x03, 0x00, 0x03, 0x03, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00, 0x03,
    0x03, 0x00, 0x00, 0x03, 0x03, 0x03, 0x03, 0x00, 0x01, 0x03, 0x02, 0x02, 0x03,
    0x01, 0x00, 0x03, 0x03, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x3E, 0x63, 0x80,
    0x80, 0x80, 0x80, 0x60, 0x3F, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xFE, 0x01, 0x01, 0x01, 0x02, 0x03, 0x3E, 0xE8, 0xF8, 0xF0, 0xD0, 0x90, 0x18,
    0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x38, 0xFF, 0x0C, 0x38, 0xE0, 0x80, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1F, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x33, 0x5F, 0x8F, 0x84, 0x05, 0x07, 0x06, 0x0C, 0x0E, 0x0E, 0x0C, 0x14, 0x34,
    0x68, 0x88, 0xD8, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x10, 0x10, 0x10,
    0xF0, 0xE0, 0x00, 0xF0, 0xF0, 0x00, 0x80, 0x80, 0x00, 0x00, 0x80, 0x80, 0x80,
    0x80, 0x00, 0x80, 0x80, 0x00, 0x80, 0x00, 0x00, 0x20, 0x38, 0x0E, 0x01, 0xC0,
    0x3F, 0xE0, 0x00, 0x00, 0x03, 0x0E, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xFF, 0xFF, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0xB6,
    0xED, 0xC0, 0xC0, 0xC0, 0xE0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA1, 0xA1, 0xA1, 0xA1,
    0xA1, 0xA1, 0xA1, 0xE1, 0xE1, 0xC1, 0xEF, 0xBB, 0x83, 0x86, 0x88, 0xB0, 0x80,
    0x80, 0x80, 0x8F, 0x90, 0x90, 0x90, 0x9F, 0x8F, 0x80, 0x9F, 0x9F, 0x87, 0x8D,
    0x98, 0x80, 0x8C, 0x9E, 0x92, 0x92, 0x9F, 0xC0, 0xC7, 0xFF, 0xB8, 0x8F, 0x80,
    0x90, 0x90, 0xC0, 0xF0, 0x8E, 0x81, 0x80, 0x81, 0x8F, 0xB8, 0xE0, 0x80, 0x80,
    0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0xFF,
);


sub init_wiring
{
    HiPi::Wiring::wiringPiSetupGpio();

    foreach my $pin ( POWER, RST, DC, LED ) {
        HiPi::Wiring::pinMode( $pin, WPI_OUTPUT );
        HiPi::Wiring::digitalWrite( $pin, WPI_LOW );
    }

    my $spi = HiPi::Device::SPI->new(
        devicename  => SPI_DEV,
        speed       => SPI_SPEED,
        bitsperword => 8,
    );
    return $spi;
}

sub init_display
{
    my ($spi) = @_;
    # Created using suggestions from 'Kuy' on Sparkfun product comments:
    # https://www.sparkfun.com/products/10168
    #
    HiPi::Wiring::digitalWrite( POWER, WPI_HIGH );
    HiPi::Wiring::digitalWrite( RST,   WPI_HIGH );
    HiPi::Wiring::delay( 5 );

    HiPi::Wiring::digitalWrite( RST, WPI_LOW );
    HiPi::Wiring::delay( 1 );

    HiPi::Wiring::digitalWrite( RST, WPI_HIGH );
    HiPi::Wiring::delay( 5 );
    send_extended_command( $spi, 0x14 ); # Set bias 1:48
    send_extended_command( $spi, 0xBC ); # Set Vop
    send_command( $spi, 0x40 ); # Set Y addr
    send_command( $spi, 0x80 ); # Set X addr
    # Doubling up, because that's what Python lib does
    send_command( $spi, 0x40 ); # Set Y addr
    send_command( $spi, 0x80 ); # Set X addr

    HiPi::Wiring::digitalWrite( LED, WPI_HIGH );
    return 1;
}

sub send_image
{
    my ($spi, $img) = @_;
    HiPi::Wiring::digitalWrite( DC, WPI_HIGH );
    my $packed_img = pack 'C*', @$img;
    $spi->transfer( $packed_img );

    return 1;
}

sub send_extended_command
{
    my ($spi, $cmd) = @_;
    say "Sending extended command: $cmd {";
    send_command( $spi, 0x21 );
    send_command( $spi, $cmd );
    send_command( $spi, 0x20 );
    send_command( $spi, 0x0C );
    say "}";
    return 1;
}

sub send_command
{
    my ($spi, $cmd) = @_;
    say "Sending command: $cmd";
    HiPi::Wiring::digitalWrite( DC, WPI_LOW );
    my $fmt_cmd = pack 'n', $cmd;
    $spi->transfer( $fmt_cmd);
    return 1;
}


{
    my $fh = init_wiring();
    init_display( $fh );
    send_image( $fh, \@PIC_OUT );
}
